/**********************************************************************************
 *  TITLE: Web Server + Manual Button control 4 Relays using ESP32 WiFi router (Real time feedback + EEPROM)
 *  Click on the following links to learn more. 
 *  YouTube Video: https://youtu.be/f9u0eUMsOfY
 *  Related Blog : https://iotcircuithub.com/esp32-home-automation-with-web-server-eeprom/
 *  
 *  This code is provided free for project purpose and fair use only.
 *  Please do mail us to techstudycell@gmail.com if you want to use it commercially.
 *  Copyrighted © by Tech StudyCell
 *  
 *  Preferences--> Aditional boards Manager URLs : 
 *  https://raw.githubusercontent.com/espressif/arduino-esp32/gh-pages/package_esp32_dev_index.json, http://arduino.esp8266.com/stable/package_esp8266com_index.json
 *  
 *  Download Board ESP32 (3.2.0) : https://github.com/espressif/arduino-esp32
 *  
 *  Download the libraries: 
 *  AceButton Library (1.10.1): https://github.com/bxparks/AceButton
 *
 * Here are the steps to use the code:
 *
 * Install Dependencies: Ensure the ESP32 board is added to Arduino IDE and install the WiFi and WebServer libraries if not pre-installed.
 * Configure Pins: Connect the four relay modules to GPIO pins 23, 19, 18 and 5 (active low).
   Connect four pushbuttons to GPIO pins 13, 12, 14 and 27.
 * Upload Code:Copy the code into the Arduino IDE, connect your ESP32, and upload the code.
 * Connect to ESP32 Wi-Fi: On a Wi-Fi-enabled device, connect to the router WiFi network with the SSID and password.
 * Access the Web Interface: Open a web browser and go to the fixed IP address 192.168.1.100 (http://192.168.1.100).
   You will see buttons to control each relay, a "EEPROM Restore" toggle button to active/deactivate EEPROM.
 * Control Relays: Use the buttons on the page to turn relays ON/OFF.

 **********************************************************************************/

#include <WiFi.h>
#include <WebServer.h>
#include <AceButton.h>
#include <EEPROM.h>
using namespace ace_button;

// Relay & Button pins
const int relayPins[] = {23, 19, 18, 5};
const int buttonPins[] = {13, 12, 14, 27};
const int numRelays = 4;
bool relayStates[numRelays] = {false, false, false, false};
bool eepromRestoreFlag = false;

const int eepromLed = 2;   //D2

// EEPROM Addresses
#define EEPROM_SIZE 10
#define FLAG_ADDR 0
#define RELAY_ADDR 1

// Button Configs & Objects
ButtonConfig buttonConfigs[numRelays];
AceButton buttons[numRelays] = {
  AceButton(&buttonConfigs[0]),
  AceButton(&buttonConfigs[1]),
  AceButton(&buttonConfigs[2]),
  AceButton(&buttonConfigs[3])
};

// WebServer instance
WebServer server(80);

// Fixed IP configuration
IPAddress local_IP(192, 168, 1, 100);
IPAddress gateway(192, 168, 1, 1);
IPAddress subnet(255, 255, 255, 0);

// WiFi credentials
const char* ssid = "";
const char* password = "";

// ==================== Setup ====================
void setup() {
  Serial.begin(115200);
  EEPROM.begin(EEPROM_SIZE);

  // Load EEPROM flag & relay states
  eepromRestoreFlag = EEPROM.read(FLAG_ADDR);
  Serial.printf("EEPROM Flag: %d\n", eepromRestoreFlag);

  if (eepromRestoreFlag) {
    for (int i = 0; i < numRelays; i++) {
      relayStates[i] = EEPROM.read(RELAY_ADDR + i);
    }
  }

  // Connect to WiFi with fixed IP
  WiFi.config(local_IP, gateway, subnet);
  WiFi.begin(ssid, password);
  Serial.print("Connecting to WiFi");
  while (WiFi.status() != WL_CONNECTED) {
    delay(500); Serial.print(".");
  }
  Serial.println("\nConnected!");
  Serial.print("ESP IP: "); Serial.println(WiFi.localIP());

  // Initialize relay pins
  for (int i = 0; i < numRelays; i++) {
    pinMode(relayPins[i], OUTPUT);
    digitalWrite(relayPins[i], relayStates[i] ? LOW : HIGH); // Active LOW
  }

  // Initialize button pins & handlers
  for (int i = 0; i < numRelays; i++) {
    pinMode(buttonPins[i], INPUT_PULLUP);
    buttons[i].init(buttonPins[i]);
    buttonConfigs[i].setEventHandler(handleButtonEvent);
  }

  pinMode(eepromLed, OUTPUT); // Initialize LED
  digitalWrite(eepromLed, LOW);

  // Web routes
  server.on("/", handleRoot);
  server.on("/toggle", handleToggle);
  server.on("/status", handleStatus);
  server.on("/alloff", handleAllOff);
  server.on("/toggleFlag", handleToggleFlag);

  server.begin();
  Serial.println("Web Server started");
}

// ==================== Loop ====================
void loop() {
  server.handleClient();
  for (uint8_t i = 0; i < numRelays; i++) {
    buttons[i].check();
  }

  digitalWrite(eepromLed, eepromRestoreFlag);
}

// ==================== Button Handler ====================
void handleButtonEvent(AceButton* button, uint8_t eventType, uint8_t buttonState) {
  if (eventType == AceButton::kEventReleased) {
    int index = button - buttons;
    relayStates[index] = !relayStates[index];
    digitalWrite(relayPins[index], relayStates[index] ? LOW : HIGH);
    Serial.printf("Relay %d toggled by button\n", index + 1);

    if (eepromRestoreFlag) saveRelayStatesToEEPROM();
  }
}

// ==================== EEPROM Save ====================
void saveRelayStatesToEEPROM() {
  for (int i = 0; i < numRelays; i++) {
    EEPROM.write(RELAY_ADDR + i, relayStates[i]);
  }
  EEPROM.commit();
}

// ==================== Web Handlers ====================
void handleRoot() {
  String html = R"rawliteral(
<!DOCTYPE html>
<html>
<head>
  <title>ESP32 Home Automation</title>
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <style>
    body { font-family: Arial, sans-serif; margin: 0; background: #f0f0f0; }
    h2 { text-align: center; padding: 20px 0; color: #333; }
    .grid { display: grid; grid-template-columns: repeat(auto-fit, minmax(150px, 1fr)); gap: 15px; padding: 0 20px; }
    .card { background: white; padding: 20px; border-radius: 10px; box-shadow: 0 2px 5px rgba(0,0,0,0.2); text-align: center; }
    .btn { padding: 10px 20px; border: none; border-radius: 8px; font-size: 16px; cursor: pointer; width: 100%; }
    .on { background: #28a745; color: white; }
    .off { background: #dc3545; color: white; }
    .alloff-btn { margin: 20px; padding: 15px; background: #007bff; color: white; border: none; border-radius: 8px; font-size: 18px; width: calc(100% - 40px); cursor: pointer; }
    .toggle-flag { margin: 20px; padding: 15px; background: #6c757d; color: white; border: none; border-radius: 8px; font-size: 18px; width: calc(100% - 40px); cursor: pointer; }
    .footer { text-align: center; padding: 15px; font-size: 14px; color: #666; }
  </style>
</head>
<body>
  <h2>ESP32 Home Automation</h2>
  <div class="grid">
    <div class="card"><button id="btn0" class="btn" onclick="toggleRelay(0)">Relay 1</button></div>
    <div class="card"><button id="btn1" class="btn" onclick="toggleRelay(1)">Relay 2</button></div>
    <div class="card"><button id="btn2" class="btn" onclick="toggleRelay(2)">Relay 3</button></div>
    <div class="card"><button id="btn3" class="btn" onclick="toggleRelay(3)">Relay 4</button></div>
  </div>
  <button class="alloff-btn" onclick="allOff()">ALL OFF</button>
  <button class="toggle-flag" onclick="toggleFlag()">EEPROM Restore: <span id="flagState"></span></button>
  <div class="footer">Created by Tech Studycell</div>
<script>
function toggleRelay(id) {
  fetch('/toggle?relay=' + id).then(fetchStatus);
}
function allOff() {
  fetch('/alloff').then(fetchStatus);
}
function toggleFlag() {
  fetch('/toggleFlag').then(fetchStatus);
}
function fetchStatus() {
  fetch('/status')
    .then(response => response.json())
    .then(data => {
      for (let i = 0; i < 4; i++) {
        let btn = document.getElementById('btn' + i);
        if (data['relay' + i]) {
          btn.className = 'btn on';
          btn.textContent = 'Relay ' + (i + 1) + ' (ON)';
        } else {
          btn.className = 'btn off';
          btn.textContent = 'Relay ' + (i + 1) + ' (OFF)';
        }
      }
      document.getElementById('flagState').textContent = data['eepromFlag'] ? 'ON' : 'OFF';
    });
}
setInterval(fetchStatus, 2000);
fetchStatus();
</script>
</body>
</html>
)rawliteral";
  server.send(200, "text/html", html);
}

void handleToggle() {
  if (server.hasArg("relay")) {
    int relay = server.arg("relay").toInt();
    if (relay >= 0 && relay < numRelays) {
      relayStates[relay] = !relayStates[relay];
      digitalWrite(relayPins[relay], relayStates[relay] ? LOW : HIGH);
      Serial.printf("Relay %d toggled by web\n", relay + 1);

      if (eepromRestoreFlag) saveRelayStatesToEEPROM();
    }
  }
  handleStatus();
}

void handleAllOff() {
  for (int i = 0; i < numRelays; i++) {
    relayStates[i] = false;
    digitalWrite(relayPins[i], HIGH);
  }
  Serial.println("All relays turned OFF by web");

  if (eepromRestoreFlag) saveRelayStatesToEEPROM();

  handleStatus();
}

void handleToggleFlag() {
  eepromRestoreFlag = !eepromRestoreFlag;
  EEPROM.write(FLAG_ADDR, eepromRestoreFlag);
  EEPROM.commit();
  Serial.printf("EEPROM Restore Flag changed to: %d\n", eepromRestoreFlag);
  handleStatus();
}

void handleStatus() {
  String json = "{";
  for (int i = 0; i < numRelays; i++) {
    json += "\"relay" + String(i) + "\":" + (relayStates[i] ? "true" : "false") + ",";
  }
  json += "\"eepromFlag\":" + String(eepromRestoreFlag ? "true" : "false") + "}";
  server.send(200, "application/json", json);
}
